<?php

class CycleAPI {
    private $username;
    private $password;
    private $base_web;
    private $base_processing;

    public function __construct() {
        $env = CYCLE_ENV;

        if ($env === 'live') {
            $this->username = CYCLE_LIVE_USERNAME;
            $this->password = CYCLE_LIVE_PASSWORD;
        } else {
            $this->username = CYCLE_TEST_USERNAME;
            $this->password = CYCLE_TEST_PASSWORD;
        }

        $this->base_web = CYCLE_WEB_API_URL;
        $this->base_processing = CYCLE_PROCESSING_API_URL;
    }

    private function sendRequest($url, $method = 'POST', $payload = []) {
        $headers = [
            "Authorization: Basic " . base64_encode("{$this->username}:{$this->password}"),
            "Content-Type: application/json"
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        } elseif ($method === 'GET') {
            curl_setopt($ch, CURLOPT_HTTPGET, true);
        }

        $result = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        if ($error) {
            return ["error" => $error];
        }

        return json_decode($result, true);
    }

    public function getClientId() {
        $url = $this->base_web . "api/v1/user/info";
        $response = $this->sendRequest($url, 'GET');

        if (isset($response['Data']['ClientID'])) {
            return $response['Data']['ClientID'];
        } else {
            echo "<pre>❌ Failed to fetch user info:\n" . json_encode($response, JSON_PRETTY_PRINT) . "</pre>";
            return null;
        }
    }

    public function createAgent($merchantId, $payload) {
        $url = $this->base_web . "api/v1/merchant/{$merchantId}/pos/create";
        return $this->sendRequest($url, 'POST', $payload);
    }

    /**
     * ✅ Fetch all transactions for ClientID (optionally filter by POSID)
     */
    public function getClientTransactions($fromDate, $toDate, $filterPosId = null) {
        $url = $this->base_processing . "api/v1/payment/listHistoryEx";
        $page = 1;
        $allTransactions = [];

        do {
            $payload = $this->basePayload($fromDate, $toDate);
            $payload["Page"] = $page;

            $response = $this->sendRequest($url, 'POST', $payload);
            $transactions = $response["Transactions"] ?? [];

            // Optional POSID filter (locally applied)
            if ($filterPosId) {
                $transactions = array_filter($transactions, function ($tx) use ($filterPosId) {
                    return isset($tx['PosID']) && $tx['PosID'] == $filterPosId;
                });
            }

            $allTransactions = array_merge($allTransactions, $transactions);
            $page++;
        } while (!empty($transactions));

        return $allTransactions;
    }

    /**
     * ✅ Fetch transactions directly from API with embedded POSID filter
     */
    public function getAgentTransactionsByHistoryEx($posId, $fromDate, $toDate) {
        $url = $this->base_processing . "api/v1/payment/listHistoryEx";
        $page = 1;
        $allTransactions = [];

        do {
            $payload = $this->basePayload($fromDate, $toDate);
            $payload["Page"] = $page;
            $payload["ListBasicModel"] = ["POSID" => $posId];

            $response = $this->sendRequest($url, 'POST', $payload);
            $transactions = $response["Transactions"] ?? [];

            $allTransactions = array_merge($allTransactions, $transactions);
            $page++;
        } while (!empty($transactions));

        return $allTransactions;
    }

    /**
     * 📦 Base payload template used by all transaction endpoints
     */
    private function basePayload($fromDate, $toDate) {
        return [
            "Period" => [
                "From" => $fromDate,
                "To" => $toDate
            ],
            "AppFramework" => "java",
            "GMT" => 2,
            "Lang" => "en",
            "IP" => $_SERVER['REMOTE_ADDR'] ?? "127.0.0.1",
            "DeviceInfo" => [
                "PhoneManufacturer" => "Cycle",
                "PhoneModel" => "CycleAdmin",
                "DeviceID" => "CycleAdminDashboard",
                "DeviceType" => "1",
                "AppFramework" => "java",
                "YOU" => "Linux",
                "OSVersion" => "1.0",
                "AppID" => "SwirlAdmin",
                "BuildNumber" => "1.0"
            ],
            "Country" => "South Africa",
            "CountryCode" => "ZA",
            "Location" => [
                "Latitude" => 0,
                "Longitude" => 0
            ]
        ];
    }
}
